﻿//******************************************************************************
//Name          : Display.cs
//Description   : Visualization with Matplotlib
//Author        : Dr. Oliver Kluge, 90411 Nuernberg, Germany
//Copyright(C)  : Oliver Kluge 2020 
//------------------------------------------------------------------------------
//******************************************************************************


using System.Globalization;
using System.IO;
using System.Diagnostics;


namespace Display
{
    public class DisplayChart
    {
        private double[] m_xdata;
        private double[] m_ydata;
        private double[] m_y2data;
        private double[,] m_zdata;
        private int m_bins;
        private double m_min;
        private double m_max;
        private string m_xlabel;
        private string m_ylabel;
        private string m_title;

        /// <summary>
        /// constructor for Plot(), BarPlot(), DotPlot() and Scatter()
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="title"></param>
        /// <param name="xlabel"></param>
        /// <param name="ylabel"></param>
        public DisplayChart(double[] x, double[] y, string title, string xlabel, string ylabel)
        {
            m_xdata = x;
            m_ydata = y;
            m_title = title;
            m_xlabel = xlabel;
            m_ylabel = ylabel;
        }

        /// <summary>
        /// constructor for Hist()
        /// </summary>
        /// <param name="x"></param>
        /// <param name="bins"></param>
        /// <param name="min"></param>
        /// <param name="max"></param>
        /// <param name="title"></param>
        /// <param name="xlabel"></param>
        /// <param name="ylabel"></param>
        public DisplayChart(double[] x, int bins, double min, double max, string title, string xlabel, string ylabel)
        {
            m_xdata = x;
            m_bins = bins;
            m_min = min;
            m_max = max;
            m_title = title;
            m_xlabel = xlabel;
            m_ylabel = ylabel;
        }

        /// <summary>
        /// constructor for Plot_3d()
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <param name="title"></param>
        /// <param name="xlabel"></param>
        /// <param name="ylabel"></param>
        public DisplayChart(double[] x, double[] y, double[,] z, string title, string xlabel, string ylabel)
        {
            m_xdata = x;
            m_ydata = y;
            m_zdata = z;
            m_title = title;
            m_xlabel = xlabel;
            m_ylabel = ylabel;
        }

        /// <summary>
        /// constructor for Plot2()
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="y2"></param>
        /// <param name="title"></param>
        /// <param name="xlabel"></param>
        /// <param name="ylabel"></param>
        public DisplayChart(double[] x, double[] y, double[] y2, string title, string xlabel, string ylabel)
        {
            m_xdata = x;
            m_ydata = y;
            m_y2data = y2;
            m_title = title;
            m_xlabel = xlabel;
            m_ylabel = ylabel;
        }

        public void Plot()
        {
            string scriptname = "myplot.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);
            string ydata = "y = [" + m_ydata[0].ToString(CultureInfo.InvariantCulture);

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            for (int i = 1; i < m_ydata.Length; i++)
            {
                ydata += ("," + m_ydata[i].ToString(CultureInfo.InvariantCulture));
            }
            ydata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine(ydata);
            pyScript.WriteLine("plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("plt.plot(x, y, color='darkblue', linestyle='solid', linewidth=1)");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.grid(True)");
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        public void BarPlot()
        {
            string scriptname = "mybarplot.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);
            string ydata = "y = [" + m_ydata[0].ToString(CultureInfo.InvariantCulture);

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            for (int i = 1; i < m_ydata.Length; i++)
            {
                ydata += ("," + m_ydata[i].ToString(CultureInfo.InvariantCulture));
            }
            ydata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine(ydata);
            pyScript.WriteLine("plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("plt.bar(x, y, color='darkblue')");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        public void DotPlot()
        {
            string scriptname = "mydotplot.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);
            string ydata = "y = [" + m_ydata[0].ToString(CultureInfo.InvariantCulture);

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            for (int i = 1; i < m_ydata.Length; i++)
            {
                ydata += ("," + m_ydata[i].ToString(CultureInfo.InvariantCulture));
            }
            ydata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine(ydata);
            pyScript.WriteLine("plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("plt.plot(x, y, color='darkblue', linestyle='solid', linewidth=0.3, marker='o', markerfacecolor='darkblue', markersize=3)");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.grid(True)");
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        public void Plot2()
        {
            string scriptname = "myplot2.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);
            string ydata = "yi = [" + m_ydata[0].ToString(CultureInfo.InvariantCulture);
            string y2data = "ym = [" + m_y2data[0].ToString(CultureInfo.InvariantCulture);

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            for (int i = 1; i < m_ydata.Length; i++)
            {
                ydata += ("," + m_ydata[i].ToString(CultureInfo.InvariantCulture));
            }
            ydata += "]";

            for (int i = 1; i < m_y2data.Length; i++)
            {
                y2data += ("," + m_y2data[i].ToString(CultureInfo.InvariantCulture));
            }
            y2data += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine(ydata);
            pyScript.WriteLine(y2data);
            pyScript.WriteLine("plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("plt.plot(x, yi, color='darkblue', linestyle='solid', linewidth=0.0, marker='o', markerfacecolor='darkblue', markersize=3)");
            pyScript.WriteLine("plt.plot(x, ym, color='darkblue', linestyle='solid', linewidth=1)");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.grid(True)");
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        public void Hist()
        {
            string scriptname = "myhist.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine("range=({0}, {1})", m_min, m_max);
            pyScript.WriteLine("bins={0}", m_bins);
            pyScript.WriteLine("plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("plt.hist(x, bins, range, color='darkblue', histtype='bar', rwidth=0.9)");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        public void Scatter()
        {
            string scriptname = "myscatter.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);
            string ydata = "y = [" + m_ydata[0].ToString(CultureInfo.InvariantCulture);

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            for (int i = 1; i < m_ydata.Length; i++)
            {
                ydata += ("," + m_ydata[i].ToString(CultureInfo.InvariantCulture));
            }
            ydata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine(ydata);
            pyScript.WriteLine("plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("plt.scatter(x, y, color = 'darkblue', marker='x', s=3)");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // using plot_surface() for 3d-visualization produces some interpolation faults in the Spectrogram context! //
        //////////////////////////////////////////////////////////////////////////////////////////////////////////////
        public void Plot_3d()
        {
            string scriptname = "myplot3d.py";
            string xdata = "x = [" + m_xdata[0].ToString(CultureInfo.InvariantCulture);
            string ydata = "y = [" + m_ydata[0].ToString(CultureInfo.InvariantCulture);
            string zdata = "z = [";

            for (int i = 1; i < m_xdata.Length; i++)
            {
                xdata += ("," + m_xdata[i].ToString(CultureInfo.InvariantCulture));
            }
            xdata += "]";

            for (int i = 1; i < m_ydata.Length; i++)
            {
                ydata += ("," + m_ydata[i].ToString(CultureInfo.InvariantCulture));
            }
            ydata += "]";

            for (int j = 0; j < m_zdata.GetLength(0); j++)
            {
                zdata += ("[" + m_zdata[j, 0].ToString(CultureInfo.InvariantCulture));
                for (int i = 1; i < m_zdata.GetLength(1); i++)
                {
                    zdata += ("," + m_zdata[j, i].ToString(CultureInfo.InvariantCulture));
                }
                zdata += "]";
                if (j < m_zdata.GetLength(0) - 1)
                {
                    zdata += ",\n";
                }
            }
            zdata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import numpy as np");
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine("from matplotlib import cm");
            pyScript.WriteLine("from mpl_toolkits.mplot3d import Axes3D");
            pyScript.WriteLine(xdata);
            pyScript.WriteLine(ydata);
            pyScript.WriteLine("x, y = np.meshgrid(x, y)");
            pyScript.WriteLine(zdata);
            pyScript.WriteLine("z = np.array(z)");
            pyScript.WriteLine("fig = plt.figure(figsize=(6.4, 4.8)) # size in inch with dpi=100 (default)");
            pyScript.WriteLine("ax = Axes3D(fig)");
            pyScript.WriteLine("surf = ax.plot_surface(x, y, z, rstride=1, cstride=1, cmap=cm.coolwarm)");
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

        public void Heatmap()
        {
            string scriptname = "myheatmap.py";
            string zdata = "z = [";
            int samplerate = m_zdata.GetLength(1) * 2;

            // from the samplerate we conclude the range for the x-axis (frequency) and the y-axis (time)
            int x0 = 0;
            int x1 = samplerate / 2;
            int y0 = 0;
            int y1 = samplerate;

            for (int j = 0; j < m_zdata.GetLength(0); j++)
            {
                zdata += ("[" + m_zdata[j, 0].ToString(CultureInfo.InvariantCulture));
                for (int i = 1; i < m_zdata.GetLength(1); i++)
                {
                    zdata += ("," + m_zdata[j, i].ToString(CultureInfo.InvariantCulture));
                }
                zdata += "]";
                if (j < m_zdata.GetLength(0) - 1)
                {
                    zdata += ",\n";
                }
            }
            zdata += "]";

            StreamWriter pyScript = File.CreateText(scriptname);
            pyScript.WriteLine("import numpy as np");
            pyScript.WriteLine("import matplotlib.pyplot as plt");
            pyScript.WriteLine(zdata);
            pyScript.WriteLine("plt.imshow(z, cmap='hot', interpolation='none', origin='lower', extent=[{0},{1},{2},{3}])", x0, x1, y0, y1);
            pyScript.WriteLine("plt.xlabel('{0}')", m_xlabel);
            pyScript.WriteLine("plt.ylabel('{0}')", m_ylabel);
            pyScript.WriteLine("plt.title('{0}')", m_title);
            pyScript.WriteLine("plt.show()");
            pyScript.Close();

            Process p = new Process();
            p.StartInfo = new ProcessStartInfo(@"Python.exe", scriptname)
            {
                UseShellExecute = false
            };
            p.Start();
            p.WaitForExit();
        }

    }

}
